#!/usr/bin/env python
import sys
import gtk
import gnome.ui
import gnome.config
import GDK
import libglade
import string
import os.path
import locale

locale.setlocale(locale.LC_NUMERIC, 'C')


############################################################
# Settings

Orig_PHYS_X=800.0
Orig_PHYS_Y=600.0
PHYS_X=Orig_PHYS_X
PHYS_Y=Orig_PHYS_Y

LAT=0
LON=1
MAXSTRIPES=20

DEF_WAYPOINTFILE="waypoints"
wpfile=gnome.config.get_string("/xpygarmin/Prefs/waypointfile")
if not wpfile:
  wpfile=DEF_WAYPOINTFILE

minlon=-180
maxlon=180
minlat=-90
maxlat=90
data=[(0.0)]
dataaccels={}
points=[(0,0)]
pointsaccels={}
waypoints=[]

############################################################
# GUI-Objects

fname = 'xpygarmin.glade'
gtk.rc_parse("xpygarmin.rc")
# create widget tree ...
xml = libglade.GladeXML(fname)

# if there is anything that needs to be added to the UI, we can access all
# the widgets like this:
win = xml.get_widget('XPygarmin')
waypointwin = xml.get_widget('waypoints')
waypointmenu = xml.get_widget('waypoints1')
waypointmenu.set_active(gtk.FALSE)

trackwin = xml.get_widget('track')
trackmenu = xml.get_widget('track1')
trackmenu.set_active(gtk.FALSE)

archivelist = xml.get_widget('archivelist')

table1=xml.get_widget("table1")
drawarea=xml.get_widget("drawingarea1")
hruler=xml.get_widget("hruler1")
vruler=xml.get_widget("vruler1")

#style = drawarea.get_style ().copy ()
#style.fg[gtk.STATE_NORMAL] = style.black
#drawarea.set_style (style)

latentry=xml.get_widget("latentry")
lonentry=xml.get_widget("lonentry")
altentry=xml.get_widget("altentry")


wplist=xml.get_widget("wplist")
pos_lonentry=xml.get_widget("pos_lonentry")
pos_latentry=xml.get_widget("pos_latentry")
pos_altentry=xml.get_widget("pos_altentry")
pos_nameentry=xml.get_widget("pos_nameentry")

tracklist=xml.get_widget("tracklist")

############################################################
# Data Conversion

def deg(dms):
   return string.atof(dms[0])+string.atof(dms[1])/60.0+string.atof(dms[2])/3600.0

def atodms(s, sep=" "):
  return string.split(s, sep)

def dms(deg,split=" "):
   grd = int(deg)
   min = int((deg-grd)*60)
   sec = (((deg-grd)*60)-min)*60
   return "%02d%s%02d%s%04.1f" % (grd,split,min,split,sec)

def data_to_virtual(d): # lat, lon
  vlat=PHYS_Y-(d[LAT]-minlat)*PHYS_Y/(maxlat-minlat)
  vlon=(d[LON]-minlon)*PHYS_X/(maxlon-minlon)
  return (vlon, vlat)

def virtual_to_data(v): # lon, lat
  lat=(-v[1]+PHYS_Y)*(maxlat-minlat)/PHYS_Y+minlat
  lon=v[0]*(maxlon-minlon)/PHYS_X+minlon
  return (lat, lon)

#print virtual_to_data(data_to_virtual((12,10)))
#print data_to_virtual(virtual_to_data((400,600)))

def prepare_data(data):
  points=[]
  for d in data:
    points.append(data_to_virtual(d))

  pointsaccels={}
  for i in range(0,PHYS_X/MAXSTRIPES+1):
    pointsaccels[i]=[]

  for p in points:
    stripe = int(p[LON]/MAXSTRIPES)
    pointsaccels[stripe].append(p)
  return (points, pointsaccels)
 

def dist(a, b):
  dx = abs(b[0]-a[0])
  dy = abs(b[1]-a[1])
  dist=dx*dx+dy*dy
  return dist

def nearest_point(v):
  minpt = points[0]
  mindist = 10000000
  try:
    accel = pointsaccels[int(v[LON]/MAXSTRIPES)]
    for p in accel:
      d = dist(p, v)
      if d < mindist:
        mindist=d
        minpt=p
  except KeyError:
    pass
  return minpt

def nearest_data(v):
  minpt = data[0]
  mindist = 100000.0
  try:
    accel = dataaccels[int((v[LON]-minlon)/dataaccels["delta"])]
    for p in accel:
      d = dist(p, v)
      if d<mindist:
        mindist = d
        minpt = p
  except KeyError:
    pass
  return minpt

def loadfile(fn):
  global minlon, maxlon, minlat, maxlat
  global data, dataaccels, points, pointsaccels
  minlon=180
  maxlon=-180
  minlat=90
  maxlat=-90
  data=[]
  tracklist.clear()
  f = open(fn, "r")

  line = f.readline()
  while len(line)<47 or line[0]=="!":
        line = f.readline()
  while (line):
	(nr, dweg, sumweg, dtime, sumtime, speed, alt, lat, lon, avgspeed) = \
          string.split(string.strip(line),"\t")

	lond=deg(atodms(lon))
	latd=deg(atodms(lat))
        tracklist.append( (nr, lat, lon, alt) )
	data.append( (latd, lond, string.atoi(alt), string.atoi(nr[:-1])) )
  	minlon = min(minlon, lond)
	maxlon = max(maxlon, lond)
	minlat = min(minlat, latd)
	maxlat = max(maxlat, latd)
	line=f.readline()

  # besser relativieren  
  minlon = minlon - 0.00005*minlon
  minlat = minlat - 0.00005*minlat
  maxlon = maxlon + 0.00005*maxlon
  maxlat = maxlat + 0.00005*maxlat

  dataaccels={}
  for i in range(0,MAXSTRIPES+1):
    dataaccels[i]=[]

  dlon = (maxlon-minlon)/MAXSTRIPES
  dataaccels["delta"] = dlon
  
  for d in data:
    stripe = int((d[LON]-minlon)/dlon)
    dataaccels[stripe].append(d)

  #print minlat, minlon, maxlat, maxlon
  hruler.set_range(minlon*100, maxlon*100, minlon*100, maxlon*100)
  vruler.set_range(maxlat*100, minlat*100, maxlat*100, minlat*100)
  (points, pointsaccels) = prepare_data(data)
  return 1

############################################################
# Main Window

def draw_waypoints(widget, gc):
  yellow = widget.get_colormap().alloc("yellow")
  blue = widget.get_colormap().alloc("blue")
  black = widget.get_colormap().alloc("black")
  font = widget.get_style().font
  for w in waypoints:
    wp=data_to_virtual((w[5], w[6])) #DLAT, DLON
    #print w, wp
    gc.foreground = yellow
    widget.draw_arc(gc, 1, wp[LAT]-3, wp[LON]-3, 6, 6, 0, 64*360)
    gc.foreground = blue
    gc.line_width = 2
    widget.draw_arc(gc, 0, wp[LAT]-4, wp[LON]-4, 8, 8, 0, 64*360)
    gc.foreground = black
    font=widget.get_style().font
    widget.draw_string(font, gc, wp[LAT]+4, wp[LON]-2, w[0])

def on_drawingarea1_expose_event(widget, event):
  global points, waypoints
  window=event.window
  gc=window.new_gc()
  style=widget.get_style()
  gc.foreground = style.bg[gtk.STATE_NORMAL]
  widget.draw_rectangle(gc, 1, event.area[0], event.area[1], event.area[2], event.area[3])
  color = widget.get_colormap().alloc("red")
  gc.foreground = color
  gc.line_width=2
  for p in points:
    widget.draw_arc(gc, 0, p[LAT]-2, p[LON]-2, 4, 4, 0, 64*360)

  widget.draw_lines(gc, points)
  draw_waypoints(widget, gc)

def on_drawingarea1_click_event(widget, event):
  window=event.window
  if event.button==1:
    if not event.state & GDK.SHIFT_MASK:
      v = (event.x, event.y)
      p = nearest_point(v)
      d = virtual_to_data(p)
      d = nearest_data(d)
      pos_latentry.set_text("%s"%dms(d[LAT]))
      pos_lonentry.set_text("%s"%dms(d[LON]))
      pos_altentry.set_text("%s"%(d[2]))
      pos_nameentry.set_text("")

oldpt = (-10, -10)
def on_drawingarea1_motion_event(widget, event):
  global oldpt
  vruler.emit("motion_notify_event", event)
  hruler.emit("motion_notify_event", event)
  v = (event.x, event.y)
  d = virtual_to_data(v)
  d = nearest_data(d)

  try:
    tracklist.select_row(d[3], 0)
    tracklist.moveto(d[3], 0)
    latentry.set_text("%s (%s)"%(dms(d[LAT]), d[LAT]))
    lonentry.set_text("%s (%s)"%(dms(d[LON]), d[LON]))
    altentry.set_text("%s"%d[2])
  except TypeError:
    pass

  p = nearest_point(v)
  gc=event.window.new_gc()
  gc.line_width=2

  if p!=oldpt:
    gc.foreground = widget.get_colormap().alloc("red")
    widget.draw_arc(gc, 0, oldpt[LAT]-2, oldpt[LON]-2, 4, 4, 0, 64*360)
    draw_waypoints(widget, gc)
  oldpt=p

  gc.foreground = widget.get_colormap().alloc("blue")
  widget.draw_arc(gc, 0, p[LAT]-2, p[LON]-2, 4, 4, 0, 64*360)

#------------------------------------------------------------
# Menu

def do_load_file(widget):
  fn = fs.get_filename()
  gnome.config.set_string("/xpygarmin/Prefs/workingdir", os.path.dirname(fn))
  loadfile(fn)
  win.set_title("XPygarmin: "+os.path.basename(fn))
  table1.queue_resize()
  fs.hide()

def on_open1_activate(widget):
  global fs
  fs = gtk.GtkFileSelection()
  fs.hide_fileop_buttons()
  wd=gnome.config.get_string("/xpygarmin/Prefs/workingdir")
  if wd:
    fs.set_filename(wd+"/")
  fs.complete("*.calc")
  fs.set_title("Load Track")
  fs.set_modal(gtk.TRUE)
  fs.ok_button.connect('clicked', do_load_file)
  fs.cancel_button.connect('clicked', do_fs_cancel)
  fs.show()


def do_generic_scale(factor):
  global points, pointsaccels, PHYS_X, PHYS_Y, oldpt
  PHYS_X=factor*Orig_PHYS_X
  PHYS_Y=factor*Orig_PHYS_Y
  oldpt=(-10,-10)
  drawarea.set_usize(PHYS_X, PHYS_Y)
  win.queue_resize()
  (points, pointsaccels) = prepare_data(data)

def on_1_1_activate(widget):
  do_generic_scale(1)
def on_1_2_activate(widget):
  do_generic_scale(0.5)
def on_1_4_activate(widget):
  do_generic_scale(0.25)
def on_1_8_activate(widget):
  do_generic_scale(0.125)
def on_8_1_activate(widget):
  do_generic_scale(8)
def on_4_1_activate(widget):
  do_generic_scale(4)
def on_2_1_activate(widget):
  do_generic_scale(2)

def on_track1_activate(widget):
  if trackmenu.active:
    trackwin.show()
  else:
    trackwin.hide()

def on_track_delete_event(widget, e):
  trackwin.hide()
  trackmenu.set_active(gtk.FALSE)
  return gtk.TRUE

def on_waypoints1_activate(widget):
  if waypointmenu.active:
    waypointwin.show()
  else:
    waypointwin.hide()

def on_waypoints_delete_event(widget, e):
  waypointwin.hide()
  waypointmenu.set_active(gtk.FALSE)
  return gtk.TRUE

def on_about1_activate(widget):
  about = gnome.ui.GnomeAbout('XPygarmin', '0.1', '(c) 2002', ['Chris Huebsch'], 'xpygarmin@firmen.huebsch-gemacht.de')
  about.show()

############################################################
# Waypoint-Window

#------------------------------------------------------------
# Waypoints

#selrow = -1
def on_wplist_select_row(widget, r, c, event):
  #global selrow
  #selrow = r
  data = wplist.get_row_data(r)
  pos_nameentry.set_text(data[0])
  pos_latentry.set_text(data[1])
  pos_lonentry.set_text(data[2])
  pos_altentry.set_text(data[3])

#def on_wplist_unselect_row(widget, r, c, event):
#  global selrow
#  selrow = -1

def on_add_clicked(widget):
  data = [pos_nameentry.get_text(), pos_latentry.get_text(), pos_lonentry.get_text(), pos_altentry.get_text(), "WP_dot", deg(atodms( pos_latentry.get_text())), deg(atodms(pos_lonentry.get_text())) ] 
  wplist.set_row_data(wplist.append(data), data)
  waypoints.append(data)
  drawarea.queue_draw()

def on_update_clicked(widget):
  selrow = wplist.selection[0]
  if selrow > -1:
    orow=selrow
    newdata = [pos_nameentry.get_text(), pos_latentry.get_text(), pos_lonentry.get_text(), pos_altentry.get_text(), "WP_dot", deg(atodms( pos_latentry.get_text())), deg(atodms(pos_lonentry.get_text()))] 

    waypoints.append(newdata)
    waypoints.remove(wplist.get_row_data(orow))

    wplist.remove(selrow)
    wplist.set_row_data(wplist.insert(orow, newdata), newdata)

    wplist.select_row(orow, 0)
    drawarea.queue_draw()

def on_del_clicked(widget):
  selrow = wplist.selection[0]
  if selrow > -1:
    waypoints.remove(wplist.get_row_data(selrow))
    drawarea.queue_draw()
    wplist.remove(selrow)

def on_clear_clicked(widget):
  global waypoints
  wplist.clear()
  waypoints=[]
  drawarea.queue_draw()

def do_fs_cancel(widget):
  fs.hide()

def savewaypoints(fn, list):
  f=open(fn, "w")
  for w in list: 
      f.write("%s	NC	%s	%s	alt=%s	symbol=%s\n"%tuple(w[0:5]))
  f.close()

fs = None
def do_save_wp(widget):
  fn = fs.get_filename()
  gnome.config.set_string("/xpygarmin/Prefs/workingdir_wp", os.path.dirname(fn))
  fs.hide()
  savewaypoints(fn, waypoints)

def on_save_clicked(widget):
  global fs
  fs = gtk.GtkFileSelection()
  wd=gnome.config.get_string("/xpygarmin/Prefs/workingdir_wp")
  if wd:
    fs.set_filename(wd+"/")
  fs.complete("*.wp")
  fs.set_title("Save Waypoints")
  fs.set_modal(gtk.TRUE)
  fs.ok_button.connect('clicked', do_save_wp)
  fs.cancel_button.connect('clicked', do_fs_cancel)
  fs.show()

def loadwaypoints(fn):
  try:
    f=open(fn, "r")
    line = f.readline()
    waypoints=[]
    while line:
      if line[0] in string.letters:
        linesplit =  string.split(string.strip(line),"\t")
        (name, NC, lat, lon) =  linesplit[0:4]
        for rest in linesplit[4:]:
          if rest[0:3]=="alt":
            alt = rest[4:]
          if rest[0:6]=="symbol":
            symbol= rest[7:]
        data = [name, lat, lon, alt, symbol, deg(atodms(lat)), deg(atodms(lon)) ] 
        waypoints.append(data)
      line = f.readline()
    f.close()
    return waypoints
  except:
    return []

fs = None
def do_load_wp(widget):
  global waypoints
  fn = fs.get_filename()
  gnome.config.set_string("/xpygarmin/Prefs/workingdir_wp", os.path.dirname(fn))
  fs.hide()
  wplist.clear()
  waypoints = loadwaypoints(fn)
  drawarea.queue_draw()
  for wp in waypoints:
      wplist.set_row_data(wplist.append(wp), wp)

def on_load_clicked(widget):
  global fs
  fs = gtk.GtkFileSelection()
  fs.hide_fileop_buttons()
  wd=gnome.config.get_string("/xpygarmin/Prefs/workingdir_wp")
  if wd:
    fs.set_filename(wd+"/")
  fs.complete("*.wp")
  fs.set_title("Load Waypoints")
  fs.set_modal(gtk.TRUE)
  fs.ok_button.connect('clicked', do_load_wp)
  fs.cancel_button.connect('clicked', do_fs_cancel)
  fs.show()
#------------------------------------------------------------
# Waypoint-Archive

fs = None

def on_archive_load_clicked(widget):
  global wparchive
  wparchive = loadwaypoints(wpfile)
  archivelist.clear()
  archivelist.freeze()
  for wp in wparchive:
    archivelist.set_row_data(archivelist.append(wp), wp)
  archivelist.thaw()
  b=gnome.ui.GnomeMessageBox("Loaded","Loaded", "OK")
  b.set_modal(gtk.TRUE)
  b.show()

def on_archive_save_clicked(widget):
  savewaypoints(wpfile, wparchive)
  b=gnome.ui.GnomeMessageBox("Saved","Saved", "OK")
  b.set_modal(gtk.TRUE)
  b.show()

#archiveselrow = -1
#def on_archivelist_select_row(widget, r, c, event):
#  global archiveselrow
#  archiveselrow = r
#
#def on_archivelist_unselect_row(widget, r, c, event):
#  global archiveselrow
#  archiveselrow = -1

def on_tocurrent_clicked(widget):
  selrow = archivelist.selection[0]
  if selrow > -1:
    data = archivelist.get_row_data(selrow)
    if wplist.find_row_from_data(data)==-1:
      waypoints.append(data)
      wplist.set_row_data(wplist.append(data), data)
      drawarea.queue_draw() 

def on_archivelist_button_press_event(widget, event):
  # double-click
  if event.type & GDK._2BUTTON_PRESS == GDK._2BUTTON_PRESS:
    on_tocurrent_clicked(widget)
  else:
    return 1

def on_toarchive_clicked(widget):
  selrow = wplist.selection[0]
  if selrow > -1:
    data = wplist.get_row_data(selrow)
    if archivelist.find_row_from_data(data)==-1:
      wparchive.append(data)
      archivelist.set_row_data(archivelist.append(data), data)


############################################################
# Tracklist

def on_tracklist_button_press_event(widget, event):
  # double-click
  if event.type & GDK._2BUTTON_PRESS == GDK._2BUTTON_PRESS:
#    point = tracklist.get_row_data()
#    highlightpos
    print tracklist.selection
  else:
    return 1



############################################################
# Main

def myquit(widget):
  gnome.config.sync()
  gtk.mainquit()

win.connect("destroy", myquit)
xml.signal_connect('on_exit1_activate', myquit)

cont=dir()
dict={}
import __main__, types
for c in cont:
  if c[:3]=="on_":
    if type(getattr(__main__, c))==types.FunctionType:
      dict[c] = getattr(__main__, c)

xml.signal_autoconnect(dict)

wparchive = loadwaypoints(wpfile)
archivelist.clear()
for wp in wparchive:
  archivelist.set_row_data(archivelist.append(wp), wp)

if len(sys.argv)==2:
	loadfile(sys.argv[1])
  	win.set_title("XPygarmin: "+sys.argv[1])

gtk.mainloop()
